<?php

declare(strict_types=1);

namespace Erlage\Photogram\Data;

use R;
use RedBeanPHP\OODBBean;

final class Query
{
    /**
     * @var string
     */
    private $table = '';

    /**
     * @var string
     */
    private $whereClause = '';

    /**
     * @var string
     */
    private $distinctClause = '';

    /**
     * @var string
     */
    private $orderByClause = '';

    /**
     * @var bool
     */
    private $isWhereClauseAppendReady = false;

    /**
     * @var bool
     */
    private $isDistinctClauseAppendReady = false;

    /**
     * @var array
     */
    private $bindings = array();

    /**
     * @return OODBBean[] 
     */
    public function select(): array
    {
        return R::find($this -> getTable(), $this -> getQuery(), $this -> getBindings());
    }

    public function selectDistinctOnlyFields(): array
    {
        return R::getAll(
            'SELECT DISTINCT ' . $this -> distinctClause . ' FROM ' . $this -> getTable() . ' WHERE ' . $this -> getQuery(),
            $this -> getBindings()
        );
    }

    /**
     * @return OODBBean|null 
     */
    public function selectOne()
    {
        return R::findOne($this -> getTable(), $this -> getQuery(), $this -> getBindings());
    }

    public function count(): string
    {
        return (string) R::count($this -> getTable(), $this -> getQuery(), $this -> getBindings());
    }

    public function delete(): int
    {
        return R::hunt($this -> getTable(), $this -> getQuery(), $this -> getBindings());
    }

    public function from(string $table): self
    {
        $this -> table = $table;

        return $this;
    }

    public function openParenthesis(): self
    {
        $this -> whereClause .= ' ( ';

        return $this;
    }

    public function closeParenthesis(): self
    {
        $this -> whereClause .= ' ) ';

        return $this;
    }

    public function or(): self
    {
        $this -> updateWhere(' OR ', 'keyword');

        return $this;
    }

    public function and(): self
    {
        $this -> updateWhere(' AND ', 'keyword');

        return $this;
    }

    public function where(string $field, string $value): self
    {
        $this -> updateWhere(" {$field} = ? ");

        $this -> bindings[] = $value;

        return $this;
    }

    public function whereDistinct(string $field): self
    {
        $this -> updateDistinct($field);

        return $this;
    }

    public function whereNot(string $field, string $value): self
    {
        $this -> updateWhere(" {$field} != ? ");

        $this -> bindings[] = $value;

        return $this;
    }

    public function whereStartsWith(string $field, string $value): self
    {
        $this -> updateWhere(" {$field} LIKE ? ");

        $this -> bindings[] = "{$value}%";

        return $this;
    }

    public function whereEndsWith(string $field, string $value): self
    {
        $this -> updateWhere(" {$field} LIKE ? ");

        $this -> bindings[] = "%{$value}";

        return $this;
    }

    public function whereContains(string $field, string $value): self
    {
        $this -> updateWhere(" {$field} LIKE ? ");

        $this -> bindings[] = "%{$value}%";

        return $this;
    }

    public function greaterThan(string $field, string $value): self
    {
        $this -> updateWhere(" {$field} > ? ");

        $this -> bindings[] = $value;

        return $this;
    }

    public function lessThan(string $field, string $value): self
    {
        $this -> updateWhere(" {$field} < ? ");

        $this -> bindings[] = $value;

        return $this;
    }

    public function in(string $field, array $values): self
    {
        if (0 == \count($values))
        {
            $this -> negate();

            return $this;
        }

        $this -> updateWhere(" {$field} IN (" . R::genSlots($values) . ') ');

        $this -> bindings = \array_merge($this -> bindings, $values);

        return $this;
    }

    public function orderByDesc(string $field): self
    {
        $this -> updateOrderBy(" ORDER BY {$field} DESC ");

        return $this;
    }

    public function orderByAsc(string $field): self
    {
        $this -> updateOrderBy(" ORDER BY {$field} ASC ");

        return $this;
    }

    public function limit(string $value): self
    {
        $this -> updateOrderBy(" LIMIT {$value} ");

        return $this;
    }

    private function getQuery(): string
    {
        return $this -> getWhereClause() . $this -> getOrderByClause();
    }

    private function negate(): self
    {
        $this -> updateWhere(' ? = ? ');

        $this -> bindings[] = '1';
        $this -> bindings[] = '0';

        return $this;
    }

    private function getWhereClause(): string
    {
        return $this -> whereClause;
    }

    private function getOrderByClause(): string
    {
        return $this -> orderByClause;
    }

    private function getBindings(): array
    {
        return $this -> bindings;
    }

    private function getTable(): string
    {
        return $this -> table;
    }

    private function updateWhere(string $append, string $type = 'predicate'): void
    {
        switch ($type)
        {
            case 'keyword':
                $this -> whereClause .= $append;

                if (' OR ' == $append)
                {
                    $this -> isWhereClauseAppendReady = false;
                }

            break;

            default:
                if ($this -> isWhereClauseAppendReady)
                {
                    $this -> and();
                }

                $this -> whereClause .= $append;

                $this -> isWhereClauseAppendReady = true;

            break;
        }
    }

    private function updateDistinct(string $field): void
    {
        if ($this -> isDistinctClauseAppendReady)
        {
            $this -> distinctClause .= ", {$field} ";
        }
        else
        {
            $this -> distinctClause .= " {$field} ";

            $this -> isDistinctClauseAppendReady = true;
        }
    }

    private function updateOrderBy(string $append): void
    {
        $this -> orderByClause .= $append;
    }
}
